/* debug.c
 *
 * Debugging console routines
 *
 * I've set this up with DEBUG defines so that release builds will not have
 * the debugging console - it will simply be compiled out. The functions will
 * still get called, they just won't do anything
 *
 * Ian Dendy
 * March 23, 2002
 */
#include "stdafx.h"
#include "debug.h"
#include <stdio.h>
#include <windows.h>
#define TRACEMAXSTRING 256
#define DEBUG2

static HANDLE console;	/* handle to our console - held internally */

#ifdef DEBUG2

/*
 * init_debug
 *
 *	We simply allocate a console that we can print debugging information
 *	to in the same manner as printf. The handle is held internally, so
 *	anybody using the routines need only call dprintf
 *
 *	Ian Dendy
 *	February 21, 2002
 */
void init_debug(void)
{
	AllocConsole();
	console = GetStdHandle(STD_OUTPUT_HANDLE); 
}

/*
 * stop_debug
 *
 *	Frees up the console
 *
 *	Ian Dendy
 *	February 21, 2002
 */
void stop_debug(void)
{
	FreeConsole();
}

/*
 * dprintf
 *
 *	Since the console handle is kept internally, the programmer
 *	need only call dprintf using the same semantics as printf and
 *	get exactly the same behavior
 *
 *	Ian Dendy
 *	February 21, 2002
 */
void dprintf( const char *format, ... )
{
    va_list args;
    char szBuffer[TRACEMAXSTRING];

    va_start(args,format);
    _vsnprintf(szBuffer, TRACEMAXSTRING, format, args);
    va_end(args);

    WriteConsole( console, szBuffer, strlen(szBuffer), 0, 0 );
}

/* On a release build, these will be the functions that get called
	in place of the normal ones. It adds a little overhead, but if
	the compiler is reasonalbe, it will likely just compile the whole
	business out.  :)  */
#else
void init_debug(void)
{
}
void stop_debug(void)
{
}
void dprintf( const char *format, ... )
{
}

#endif

